package de.davidartmann.rosa.adapter.viewholder;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.support.v4.content.ContextCompat;
import android.support.v7.app.AlertDialog;
import android.support.v7.widget.CardView;
import android.support.v7.widget.RecyclerView;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.TextView;

import com.squareup.picasso.Picasso;

import java.io.File;
import java.util.List;

import de.davidartmann.rosa.R;
import de.davidartmann.rosa.activity.ViewPagerActivity;
import de.davidartmann.rosa.adapter.MainListAdapter;
import de.davidartmann.rosa.db.async.FindAllActivePersonsFromDb;
import de.davidartmann.rosa.db.async.IncrementPositionOfPersonsFromDb;
import de.davidartmann.rosa.db.async.UpdatePersonInDb;
import de.davidartmann.rosa.model.Person;
import de.davidartmann.rosa.util.RoundedTransformation;

/**
 * ViewHolder for the active persons of the mainlist.
 * Created by david on 29.02.16.
 */
public class MainListViewholder extends RecyclerView.ViewHolder implements View.OnClickListener,
    UpdatePersonInDb.IUpdatePersonInDb, FindAllActivePersonsFromDb.IFindAllActive,
        IncrementPositionOfPersonsFromDb.IIncrementPosition, View.OnLongClickListener {

    //private static final String TAG = MainListViewholder.class.getSimpleName();

    private ImageView imageViewPicture;
    private TextView textViewName;
    private TextView textViewAddress;
    private TextView textViewPosition;
    private TextView textViewPrice;
    private Context mContext;
    private Person mPerson;
    private CardView cardView;
    private int mNumerOfPersons;
    private int mOldPos;
    private int mNewPos;
    private IMainListViewholder iMainListViewholder;
    private int mBgColor;
    private MainListAdapter mMainListAdapter;

    public MainListViewholder(View itemView, Context context, int numberOfPersons,
                              IMainListViewholder iMainListViewholder,
                              MainListAdapter mainListAdapter) {
        super(itemView);
        this.mContext = context;
        mNumerOfPersons = numberOfPersons;
        imageViewPicture = (ImageView) itemView.findViewById(R.id.listview_viewholder_picture);
        textViewName = (TextView) itemView.findViewById(R.id.listview_viewholder_name);
        textViewAddress = (TextView) itemView.findViewById(R.id.listview_viewholder_address);
        textViewPosition = (TextView) itemView.findViewById(R.id.listview_viewholder_position);
        textViewPrice = (TextView) itemView.findViewById(R.id.listview_viewholder_price);
        cardView = (CardView) itemView.findViewById(R.id.listview_viewholder_cardview);
        itemView.setOnClickListener(this);
        itemView.setOnLongClickListener(this);
        textViewPosition.setOnClickListener(this);
        this.iMainListViewholder = iMainListViewholder;
        mMainListAdapter = mainListAdapter;
    }

    public void assignData(Person person) {
        this.mPerson = person;
        textViewAddress.setText(person.getAddress());
        textViewName.setText(person.getName());
        textViewPosition.setText(String.valueOf(person.getPosition()));
        textViewPrice.setText(String.format("%s%s", person.getPrice(),
                mContext.getString(R.string.whitespace_eurosign)));
        if (person.getCategory() == 0) {
            mBgColor = ContextCompat.getColor(mContext, R.color.colorSure);
        } else if (person.getCategory() == 1) {
            mBgColor = ContextCompat.getColor(mContext, R.color.colorAlmostSure);
        } else if (person.getCategory() == 2) {
            mBgColor = ContextCompat.getColor(mContext, R.color.colorUnsure);
        }
        cardView.setCardBackgroundColor(mBgColor);
        File file = null;
        if (person.getPictureUrl() != null) {
            file = new File(person.getPictureUrl());
        }
        Picasso.with(mContext)
                .load(file)
                .transform(new RoundedTransformation(0, R.dimen.imageview_list_dimen))
                .resizeDimen(R.dimen.imageview_list_dimen, R.dimen.imageview_list_dimen)
                .placeholder(R.drawable.user)
                .error(R.drawable.user)
                .centerCrop()
                .into(imageViewPicture);
        //Log.d(TAG, "position=" + textViewPosition.getText().toString());
    }

    @Override
    public void onClick(View v) {
        switch (v.getId()) {
            case R.id.listview_viewholder_cardview:
                if (!mMainListAdapter.isLongClicked()) {
                    Intent intent = new Intent(mContext, ViewPagerActivity.class);
                    intent.putExtra("person", mPerson);
                    //TODO: if you want to disable the backstack,
                    // implement interface for telling mainlistactivity to finish,
                    // when this is fired
                    mContext.startActivity(intent);
                } else {
                    iMainListViewholder.onClicked(getAdapterPosition(), v, mPerson, mBgColor);
                }
                break;
            case R.id.listview_viewholder_position:
                showPositionChooserDialog();
                break;
        }
    }

    private void showPositionChooserDialog() {
        @SuppressLint("InflateParams") View view = LayoutInflater.from(mContext)
                .inflate(R.layout.dialog_position_chooser, null);
        AlertDialog.Builder builder = new AlertDialog.Builder(mContext);
        builder.setTitle(mContext.getString(R.string.Neue_Position));
        final EditText editTextNewPos =
                (EditText) view.findViewById(R.id.dialog_position_chooser_edittext);
        builder.setView(view);
        builder.setPositiveButton(mContext.getString(R.string.Speichern),
                new DialogInterface.OnClickListener() {
                    @Override
                    public void onClick(DialogInterface dialog, int which) {
                        mNewPos = Integer.valueOf(editTextNewPos.getText().toString());
                        if (mNewPos > 0 && mNewPos <= mNumerOfPersons) {
                            mOldPos = mPerson.getPosition();
                            if (mNewPos != mOldPos) {
                                mPerson.setPosition(mNewPos);
                                new FindAllActivePersonsFromDb(mContext, MainListViewholder.this)
                                        .execute();
                            }
                        }
                        dialog.dismiss();
                    }
                });
        builder.setNegativeButton(mContext.getString(R.string.Abbrechen),
                new DialogInterface.OnClickListener() {
                    @Override
                    public void onClick(DialogInterface dialog, int which) {
                        dialog.dismiss();
                    }
                });
        builder.show();
    }

    @Override
    public void onUpdate(Person person) {
        mPerson = person;
        assignData(person);
        iMainListViewholder.onIncrementFinished();
    }

    @Override
    public void onFindAllActive(List<Person> persons) {
        new IncrementPositionOfPersonsFromDb(mContext, this, persons, mOldPos, mNewPos).execute();
    }

    @Override
    public void onIncremented(List<Person> persons) {
        new UpdatePersonInDb(mContext, this).execute(mPerson);
    }

    @Override
    public boolean onLongClick(View v) {
        iMainListViewholder.onLongClicked(getAdapterPosition(), v, mPerson);
        mMainListAdapter.setmIsLongClicked(true);
        return true;
    }

    public interface IMainListViewholder {
        void onIncrementFinished();
        void onLongClicked(int position, View view, Person person);
        void onClicked(int position, View view, Person person, int bgColor);
    }
}
